import { uploadToHuggingFace } from "@gradio/utils";
import { Gradio } from "@gradio/utils";
export const format_chat_for_sharing = async (chat) => {
    let messages = await Promise.all(chat.map(async (message) => {
        if (message.role === "system")
            return "";
        let speaker_emoji = message.role === "user" ? "😃" : "🤖";
        let html_content = "";
        if (message.type === "text") {
            const regexPatterns = {
                audio: /<audio.*?src="(\/file=.*?)"/g,
                video: /<video.*?src="(\/file=.*?)"/g,
                image: /<img.*?src="(\/file=.*?)".*?\/>|!\[.*?\]\((\/file=.*?)\)/g
            };
            html_content = message.content;
            for (let [_, regex] of Object.entries(regexPatterns)) {
                let match;
                while ((match = regex.exec(message.content)) !== null) {
                    const fileUrl = match[1] || match[2];
                    const newUrl = await uploadToHuggingFace(fileUrl, "url");
                    html_content = html_content.replace(fileUrl, newUrl);
                }
            }
        }
        else {
            if (!message.content.value)
                return "";
            const url = message.content.component === "video"
                ? message.content.value?.video.path
                : message.content.value;
            const file_url = await uploadToHuggingFace(url, "url");
            if (message.content.component === "audio") {
                html_content = `<audio controls src="${file_url}"></audio>`;
            }
            else if (message.content.component === "video") {
                html_content = file_url;
            }
            else if (message.content.component === "image") {
                html_content = `<img src="${file_url}" />`;
            }
        }
        return `${speaker_emoji}: ${html_content}`;
    }));
    return messages.join("\n");
};
const redirect_src_url = (src, root) => src.replace('src="/file', `src="${root}file`);
function get_component_for_mime_type(mime_type) {
    if (!mime_type)
        return "file";
    if (mime_type.includes("audio"))
        return "audio";
    if (mime_type.includes("video"))
        return "video";
    if (mime_type.includes("image"))
        return "image";
    return "file";
}
function convert_file_message_to_component_message(message) {
    const _file = Array.isArray(message.file) ? message.file[0] : message.file;
    return {
        component: get_component_for_mime_type(_file?.mime_type),
        value: message.file,
        alt_text: message.alt_text,
        constructor_args: {},
        props: {}
    };
}
export function normalise_messages(messages, root) {
    if (messages === null)
        return messages;
    const thought_map = new Map();
    return messages
        .map((message, i) => {
        let normalized = typeof message.content === "string"
            ? {
                role: message.role,
                metadata: message.metadata,
                content: redirect_src_url(message.content, root),
                type: "text",
                index: i,
                options: message.options
            }
            : "file" in message.content
                ? {
                    content: convert_file_message_to_component_message(message.content),
                    metadata: message.metadata,
                    role: message.role,
                    type: "component",
                    index: i,
                    options: message.options
                }
                : { type: "component", ...message };
        // handle thoughts
        const { id, title, parent_id } = message.metadata || {};
        if (parent_id) {
            const parent = thought_map.get(String(parent_id));
            if (parent) {
                const thought = { ...normalized, children: [] };
                parent.children.push(thought);
                if (id && title) {
                    thought_map.set(String(id), thought);
                }
                return null;
            }
        }
        if (id && title) {
            const thought = { ...normalized, children: [] };
            thought_map.set(String(id), thought);
            return thought;
        }
        return normalized;
    })
        .filter((msg) => msg !== null);
}
export function normalise_tuples(messages, root) {
    if (messages === null)
        return messages;
    const msg = messages.flatMap((message_pair, i) => {
        return message_pair.map((message, index) => {
            if (message == null)
                return null;
            const role = index == 0 ? "user" : "assistant";
            if (typeof message === "string") {
                return {
                    role: role,
                    type: "text",
                    content: redirect_src_url(message, root),
                    metadata: { title: null },
                    index: [i, index]
                };
            }
            if ("file" in message) {
                return {
                    content: convert_file_message_to_component_message(message),
                    role: role,
                    type: "component",
                    index: [i, index]
                };
            }
            return {
                role: role,
                content: message,
                type: "component",
                index: [i, index]
            };
        });
    });
    return msg.filter((message) => message != null);
}
export function is_component_message(message) {
    return message.type === "component";
}
export function is_last_bot_message(messages, all_messages) {
    const is_bot = messages[messages.length - 1].role === "assistant";
    const last_index = messages[messages.length - 1].index;
    // use JSON.stringify to handle both the number and tuple cases
    // when msg_format is tuples, last_index is an array and when it is messages, it is a number
    const is_last = JSON.stringify(last_index) ===
        JSON.stringify(all_messages[all_messages.length - 1].index);
    return is_last && is_bot;
}
export function group_messages(messages, msg_format) {
    const groupedMessages = [];
    let currentGroup = [];
    let currentRole = null;
    for (const message of messages) {
        if (!(message.role === "assistant" || message.role === "user")) {
            continue;
        }
        if (message.role === currentRole) {
            currentGroup.push(message);
        }
        else {
            if (currentGroup.length > 0) {
                groupedMessages.push(currentGroup);
            }
            currentGroup = [message];
            currentRole = message.role;
        }
    }
    if (currentGroup.length > 0) {
        groupedMessages.push(currentGroup);
    }
    return groupedMessages;
}
export async function load_components(component_names, _components, load_component) {
    let names = [];
    let components = [];
    component_names.forEach((component_name) => {
        if (_components[component_name] || component_name === "file") {
            return;
        }
        const variant = component_name === "dataframe" ? "component" : "base";
        const { name, component } = load_component(component_name, variant);
        names.push(name);
        components.push(component);
        component_name;
    });
    const loaded_components = await Promise.all(components);
    loaded_components.forEach((component, i) => {
        _components[names[i]] = component.default;
    });
    return _components;
}
export function get_components_from_messages(messages) {
    if (!messages)
        return [];
    let components = new Set();
    messages.forEach((message) => {
        if (message.type === "component") {
            components.add(message.content.component);
        }
    });
    return Array.from(components);
}
export function get_thought_content(msg, depth = 0) {
    let content = "";
    const indent = "  ".repeat(depth);
    if (msg.metadata?.title) {
        content += `${indent}${depth > 0 ? "- " : ""}${msg.metadata.title}\n`;
    }
    if (typeof msg.content === "string") {
        content += `${indent}  ${msg.content}\n`;
    }
    const thought = msg;
    if (thought.children?.length > 0) {
        content += thought.children
            .map((child) => get_thought_content(child, depth + 1))
            .join("");
    }
    return content;
}
export function all_text(message) {
    if (Array.isArray(message)) {
        return message
            .map((m) => {
            if (m.metadata?.title) {
                return get_thought_content(m);
            }
            return m.content;
        })
            .join("\n");
    }
    if (message.metadata?.title) {
        return get_thought_content(message);
    }
    return message.content;
}
export function is_all_text(message) {
    return ((Array.isArray(message) &&
        message.every((m) => typeof m.content === "string")) ||
        (!Array.isArray(message) && typeof message.content === "string"));
}
